// src/components/CartPage.jsx
import React, { useEffect, useState } from "react";
import { useNavigate } from "react-router-dom";
import axios from "axios";

import "./CartPage.css";

const CartPage = () => {
  const [cart, setCart] = useState([]);
  const [loadingCheckout, setLoadingCheckout] = useState(false);

  const navigate = useNavigate();


  useEffect(() => {
    document.title = "React Shopping Cart - Checkout";
    const storedCart = JSON.parse(localStorage.getItem("cart")) || [];
    setCart(storedCart);
  }, []);

  // Helper: update localStorage + Navbar badge
  const syncCart = (updatedCart) => {
    setCart(updatedCart);
    localStorage.setItem("cart", JSON.stringify(updatedCart));
    const totalItems = updatedCart.reduce((sum, item) => sum + item.quantity, 0);
    window.dispatchEvent(new CustomEvent("cartUpdated", { detail: totalItems }));
  };

  const updateQuantity = (id, delta) => {
    const updated = cart.map((item) =>
      item.id === id ? { ...item, quantity: Math.max(1, item.quantity + delta) } : item
    );
    syncCart(updated);
  };

  const removeItem = (id) => {
    const updated = cart.filter((item) => item.id !== id);
    syncCart(updated);
  };


 const checkout = async () => {
  const cart = JSON.parse(localStorage.getItem("cart")) || [];

  if (cart.length === 0) return;

  setLoadingCheckout(true); // show spinner

  try {
    // Simulate 2-second delay before sending request
    await new Promise((resolve) => setTimeout(resolve, 2000));

    const response = await axios.post("http://localhost/react-shopping-cart-api/cart-checkout-api.php", { cart });
    const data = response.data;

    if (data.success) {
      // Clear cart and navigate to Thank You page
      localStorage.removeItem("cart");
      window.dispatchEvent(new CustomEvent("cartUpdated", { detail: 0 }));
      setCart([]);
      navigate("/thankyou");
    }
    // If checkout fails, do nothing silently
  } catch (error) {
    console.error("Checkout error:", error);
    // silently fail, do nothing
  } finally {
    setLoadingCheckout(false); // hide spinner
  }
};

  const subtotal = cart.reduce((acc, item) => acc + item.price * item.quantity, 0);
  const shipping = cart.length > 0 ? 3.0 : 0;
  const total = subtotal + shipping;

  // Show empty message if cart is 0
  if (cart.length === 0) {
    return (
      <h1
        style={{
          color: "#81c408",
          textAlign: "center",
          marginTop: "110px",
          backgroundColor: "#f7f7f7",
          padding: "20px",
          borderRadius: "8px",
          height: "100%"
        }}
      >
        Your cart is empty!
      </h1>
    );
  }

  return (
    <div className="cart-page">
      <div className="cart-container">
        <div className="clear-cart-wrapper">
          <button className="clear-cart-btn" onClick={() => {
            localStorage.removeItem("cart");
            setCart([]);
            window.dispatchEvent(new CustomEvent("cartUpdated", { detail: 0 }));
          }}><img src="/img/trash.svg" className="img-trash" alt="trash" />
            Clear Cart
          </button>
        </div>
        {/* Cart Table */}
        <table className="cart-table">
          <thead>
            <tr>
              <th>Products</th>
              <th>Name</th>
              <th>Price</th>
              <th>Quantity</th>
              <th>Total</th>
              <th>Handle</th>
            </tr>
          </thead>
          <tbody>
            {cart.map((item) => (
              <tr key={item.id}>
                <td><img src={item.image} alt={item.name} /></td>
                <td>{item.name}</td>
                <td>${item.price.toFixed(2)}</td>
                <td>
                  <div className="quantity-control">
                    <button className="quantity-btn" onClick={() => updateQuantity(item.id, -1)}>
                      <i className="fa fa-minus"></i>
                    </button>
                    <input type="text" className="quantity-input" value={item.quantity} readOnly />
                    <button className="quantity-btn" onClick={() => updateQuantity(item.id, 1)}>
                      <i className="fa fa-plus"></i>
                    </button>
                  </div>
                </td>
                <td>${(item.price * item.quantity).toFixed(2)}</td>
                <td>
                  <button className="remove-btn" onClick={() => removeItem(item.id)}>
                    <i className="fa fa-times"></i>
                  </button>
                </td>
              </tr>
            ))}
          </tbody>
        </table>

        {/* Cart Summary */}
        <div className="cart-layout">

          <div className="cart-summary">
            <h2>Cart Total</h2>
            <div className="summary-row">
              <h5>Subtotal:</h5>
              <p>${subtotal.toFixed(2)}</p>
            </div>
            <div className="summary-row">
              <h5>Shipping:</h5>
              <p>${shipping.toFixed(2)}</p>
            </div>
            <p style={{ textAlign: "right", marginBottom: "1rem" }}>
              Shipping to Ukraine
            </p>
            <div className="summary-total">
              <h5>Total:</h5>
              <p>${total.toFixed(2)}</p>
            </div>
            <button
              className="checkout-btn"
              onClick={checkout}
              disabled={loadingCheckout}
              style={{
                display: "flex",
                alignItems: "center",
                justifyContent: "center",
                gap: "8px", // space between spinner and text
              }}
            >
              {loadingCheckout && (
                <img
                  src="img/loader.svg"
                  alt="Loading..."
                  style={{ width: "20px", height: "20px" }}
                />
              )}
              {loadingCheckout ? "Processing..." : "Proceed Checkout"}
            </button>


          </div>
        </div>
      </div>
    </div>
  );
};

export default CartPage;
