import { useState, useEffect, Fragment } from "react";
import axios from "axios";
import AddUserForm from "./components/forms/AddUserForm";
import EditUserForm from "./components/forms/EditUserForm";
import UserTable from "./components/tables/UserTable";
import { toast, ToastContainer } from "react-toastify";
import "react-toastify/dist/ReactToastify.css";
import { SERVER_SIDE_API_ROOT } from "./config";

const API_URL = `${SERVER_SIDE_API_ROOT}/users`;

const App = () => {
  const [users, setUsers] = useState([]);
  const [currentUser, setCurrentUser] = useState({ id: null, name: "", username: "" });
  const [editing, setEditing] = useState(false);

  const fetchUsers = async () => {
    const res = await axios.get(API_URL);
    setUsers(res.data);
  };

  useEffect(() => {
    fetchUsers();
  }, []);

  const addUser = async (user) => {
    const res = await axios.post(API_URL, user);
    setUsers([...users, res.data]);
  };

  const deleteUser = async (id) => {
  toast.info(
    <div>
      <p>Are you sure you want to delete this user?</p>
      <div
        style={{
          display: "flex",
          justifyContent: "center",
          gap: "10px",
          marginTop: "10px",
        }}>
        <button
          onClick={async () => {
            try {
              await axios.delete(`${API_URL}/${id}`);
              setUsers(users.filter((u) => u.id !== id));
              toast.dismiss(); 
            } catch (err) {
              toast.dismiss();
              console.error("Error deleting user:", err);
            }
          }}
          style={{
            background: "#232323",
            color: "#fff",
            border: "none",
            padding: "6px 14px",
            borderRadius: "6px",
            cursor: "pointer",
          }}>
          Yes
        </button>

        <button
          onClick={() => toast.dismiss()}
          style={{
            background: "#ccc",
            color: "#000",
            border: "none",
            padding: "6px 14px",
            borderRadius: "6px",
            cursor: "pointer",
          }}>
          No
        </button>
      </div>
    </div>,
    {
      autoClose: false,
      closeOnClick: false,
      draggable: false,
    }
  );
};

  const updateUser = async (id, updatedUser) => {
    const res = await axios.put(`${API_URL}/${id}`, updatedUser);
    setUsers(users.map((u) => (u.id === id ? res.data : u)));
    setEditing(false);
  };

  const editRow = (user) => {
    setEditing(true);
    setCurrentUser(user);
  };

  return (
    <div className="container">
     <ToastContainer position="top-center" />
      <h1>React + Node + MySQL CRUD</h1>
      <div className="flex-row">
        <div className="flex-large">
          {editing ? (
            <Fragment>
              <h2>Edit user</h2>
              <EditUserForm
                setEditing={setEditing}
                currentUser={currentUser}
                updateUser={updateUser}/>
            </Fragment>
          ) : (
            <Fragment>
              <h2>Add user</h2>
              <AddUserForm addUser={addUser} />
            </Fragment>
          )}
        </div>
        <div className="flex-large">
          <h2>View users</h2>
          <UserTable users={users} editRow={editRow} deleteUser={deleteUser} />
        </div>
      </div>
    </div>
  );
};

export default App;
