import React, { useState } from "react";
import axios from "axios";
import { useNavigate } from "react-router-dom";
import SERVER_SIDE_API_ROOT from "../config";
import "../styles/style.css";

const RegisterForm = () => {
  const [formData, setFormData] = useState({
    username: "",
    email: "",
    password: "",
    confirmPassword: "",
  });
  const [errorMessage, setErrorMessage] = useState("");
  const [successMessage, setSuccessMessage] = useState("");
  const navigate = useNavigate();

  const handleSubmit = async (e) => {
  e.preventDefault();

  const { username, email, password, confirmPassword } = formData;

  if (!username || !email || !password || !confirmPassword) {
    setErrorMessage("Please fill in all fields");
    setSuccessMessage("");
    return;
  }

  const emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
  if (!emailRegex.test(email)) {
    setErrorMessage("Please enter a valid email address");
    return;
  }

  if (password.length < 6) {
    setErrorMessage("Password must be at least 6 characters");
    return;
  }

  if (password !== confirmPassword) {
    setErrorMessage("Passwords do not match");
    setSuccessMessage("");
    return;
  }

  try {
    const res = await axios.post(
      `${SERVER_SIDE_API_ROOT}/registration-action.php`,
      { username, email, password },
      { headers: { "Content-Type": "application/json" } }
    );

    console.log("Server response:", res.data);

    if (res.data.status === "success") {
      setSuccessMessage(res.data.message);
      setErrorMessage("");

      setFormData({
        username: "",
        email: "",
        password: "",
        confirmPassword: "",
      });

      setTimeout(() => {
        navigate("/login");
      }, 2000);
    } else {
      setErrorMessage(res.data.message || "Registration failed");
      setSuccessMessage("");
    }
  } catch (err) {
    console.error("Axios Error:", err);
    setErrorMessage("Server error or CORS issue");
    setSuccessMessage("");
  }
};


return (
  <div className="admin-wrapper">
    <div
      className="card-container"
      style={{ maxWidth: "400px", margin: "95px auto" }}
    >
    <h2>User Registration</h2>
    <form onSubmit={handleSubmit}>
      <div>
        <label>Username:</label>
        <input
          value={formData.username}
          onChange={(e) =>
            setFormData({ ...formData, username: e.target.value })
          }            
        />
      </div>
      <div>
        <label>Email:</label>
        <input
          type="email"
          value={formData.email}
          onChange={(e) =>
            setFormData({ ...formData, email: e.target.value })
          }            
        />
      </div>
      <div>
        <label>Password:</label>
        <input
          type="password"
          value={formData.password}
          onChange={(e) =>
            setFormData({ ...formData, password: e.target.value })
          }            
        />
      </div>
      <div>
        <label>Confirm Password:</label>
        <input
          type="password"
          value={formData.confirmPassword}
          onChange={(e) =>
            setFormData({
              ...formData,
              confirmPassword: e.target.value,
            })
          }            
        />
      </div>
      {errorMessage && (
        <div className="alert alert-danger" role="alert">
          {errorMessage}
        </div>
      )}
      {successMessage && (
        <div className="alert alert-success" role="alert">
          {successMessage}
        </div>
      )}
      <button type="submit">Register</button>
      <p style={{ textAlign: "center", marginTop: "10px" }}>
        Already have an account?{" "}
        <a href="/login" style={{ color: "#232323", fontWeight: "600",textDecoration: "none" }}>
          Login here
        </a>
      </p>
    </form>
    </div>
  </div>
);
};

export default RegisterForm;
