import { useEffect, useState } from "react";
import { GoogleMap, useJsApiLoader } from "@react-google-maps/api";
import LocationMarker from "./LocationMarker";

export default function MapContainerComponent({ searchQuery }) {
  const [map, setMap] = useState(null);
  const [userLocation, setUserLocation] = useState(null);
  const [searchLocation, setSearchLocation] = useState(null);

  const { isLoaded } = useJsApiLoader({
    googleMapsApiKey: "YOUR API KEY", // API key
    libraries: ["places"],
  });

  useEffect(() => {
    if (navigator.geolocation) {
      const watchId = navigator.geolocation.watchPosition(
        (pos) => {
          const newLoc = {
            lat: pos.coords.latitude,
            lng: pos.coords.longitude,
          };
          setUserLocation(newLoc);
          if (map && !searchLocation) {
            map.setCenter(newLoc);
            map.setZoom(13);
          }
        },
        (err) => console.error("Location error:", err),
        { enableHighAccuracy: true, maximumAge: 1000 }
      );
      return () => navigator.geolocation.clearWatch(watchId);
    } else {
      console.error("Geolocation not supported");
    }
  }, [map, searchLocation]);

  useEffect(() => {
    if (!searchQuery || !window.google || !map) return;
    const geocoder = new window.google.maps.Geocoder();
    geocoder.geocode({ address: searchQuery }, (results, status) => {
      if (status === "OK" && results[0]) {
        const loc = results[0].geometry.location;
        const newSearchLoc = { lat: loc.lat(), lng: loc.lng() };
        setSearchLocation(newSearchLoc);
        if (userLocation) {
          const bounds = new window.google.maps.LatLngBounds();
          bounds.extend(userLocation);
          bounds.extend(newSearchLoc);
          map.fitBounds(bounds);
        } else {
          map.setCenter(newSearchLoc);
          map.setZoom(12);
        }
      } else {
        console.warn("Location not found for:", searchQuery);
      }
    });
  }, [searchQuery, map, userLocation]);

  const zoomToLocation = (loc) => {
    if (!map || !loc) return;
    map.panTo(loc);
    map.setZoom(15);
  };

  return (
    <div className="map-container">
      {isLoaded && (
        <GoogleMap
          mapContainerStyle={{ width: "100%", height: "100vh" }}
          center={userLocation || { lat: 20.5937, lng: 78.9629 }}
          zoom={userLocation ? 13 : 5}
          onLoad={setMap}
          options={{
            streetViewControl: false,
            mapTypeControl: false,
            fullscreenControl: false,
          }}>
          <LocationMarker
            position={userLocation}
            title="Your Location"
            onClick={() => zoomToLocation(userLocation)}
          />
          <LocationMarker
            position={searchLocation}
            title="Tracked Location"
            onClick={() => zoomToLocation(searchLocation)}
          />
        </GoogleMap>
      )}
      {userLocation && (
        <button
          className="floating-btn"
          onClick={() => zoomToLocation(userLocation)}>
          📍 My Location
        </button>
      )}
    </div>
  );
}