import { useState } from "react";
import axios from "axios";
import SERVER_SIDE_API_ROOT from "../../config";
import FilePreview from "./FilePreview";
import UploadBox from "./UploadBox";
import "../../public/assets/css/style.css";

const FileUpload = () => {
  const [files, setFiles] = useState([]);
  const [dragActive, setDragActive] = useState(false);
  const [uploading, setUploading] = useState(false);
  const [errorMsg, setErrorMsg] = useState("");

  const allowedExtensions = ["jpg", "jpeg", "png", "gif", "pdf", "doc", "docx", "txt"];
  const MAX_FILE_SIZE = 2 * 1024 * 1024;

  const uploadFiles = async (fileList) => {
    setErrorMsg("");
    const safeFiles = [];
    const rejectedFiles = [];

    fileList.forEach((file) => {
      const ext = file.name.split(".").pop().toLowerCase();
      if (!allowedExtensions.includes(ext)) return rejectedFiles.push("Invalid file type");
      if (file.size > MAX_FILE_SIZE) return rejectedFiles.push("Maximum file size is 2MB.");
      if (file.size <= 0) return rejectedFiles.push("Empty file");
      safeFiles.push(file);
    });

    if (rejectedFiles.length > 0) {
      setErrorMsg(rejectedFiles[0]);
      setUploading(false);
      return;
    }
    
    if (!safeFiles.length) return;

    const formData = new FormData();
    safeFiles.forEach((file) => formData.append("files[]", file));
    setUploading(true);

    const delay = new Promise((resolve) => setTimeout(resolve, 800));
    try {
      const res = await Promise.all([
        axios.post(`${SERVER_SIDE_API_ROOT}/file-upload.php`, formData, {
          headers: { "Content-Type": "multipart/form-data" },
        }),
        delay,
      ]);
      const uploadedFiles = res[0].data.files.filter((f) => f.status === "uploaded");
      setFiles((prev) => [...prev, ...uploadedFiles.map(f => safeFiles.find(sf => sf.name === f.name))]);
    } catch {
      setErrorMsg("Server error — please try again later.");
    }
    setUploading(false);
  };

  const handleDrop = async (e) => {
    e.preventDefault();
    setDragActive(false);
    await uploadFiles(Array.from(e.dataTransfer.files));
  };

  return (
    <div className="upload-wrapper">
      <UploadBox
        dragActive={dragActive}
        uploading={uploading}
        errorMsg={errorMsg}
        handleDrop={handleDrop}
        setDragActive={setDragActive}
      >
      </UploadBox>
      <FilePreview files={files} uploading={uploading} />
    </div>
  );
};

export default FileUpload;
