import { useEffect, useState, useCallback } from "react";
import axios from "axios";
import {
    useReactTable,
    getCoreRowModel,
    getSortedRowModel,
    flexRender,
} from "@tanstack/react-table";
import "./TanStack.css";

const TanStack = () => {
    const [data, setData] = useState([]);
    const [search, setSearch] = useState("");
    const [page, setPage] = useState(1);
    const [limit, setLimit] = useState(5);
    const [total, setTotal] = useState(0);
    const [sortBy, setSortBy] = useState("id");
    const [order, setOrder] = useState("asc");

    const fetchData = useCallback(async () => {
        try {
            const res = await axios.get("http://localhost/react-product-table-api/products.php", {
                params: { search, page, limit, sortBy, order },
            });

            const fetchedData = res.data.data || res.data.products || [];
            setData(fetchedData);
            setTotal(res.data.total || 0);
        } catch (err) {
            console.error("Error fetching data:", err);
        }
    }, [search, page, limit, sortBy, order]);

    useEffect(() => {
        fetchData();
    }, [fetchData]);

    const columns = [
        {
            header: "S.No",
            accessorKey: "sno",
            cell: (info) => (page - 1) * limit + info.row.index + 1,
        },
        {
            header: "Title",
            accessorKey: "title",
            cell: (info) => info.getValue(),
        },
        {
            header: "Price",
            accessorKey: "price",
            cell: (info) => `$${info.getValue()}`,
        },
        {
            header: "Category",
            accessorKey: "category",
            cell: (info) => info.getValue(),
        },
    ];

    const table = useReactTable({
        data,
        columns,
        getCoreRowModel: getCoreRowModel(),
        getSortedRowModel: getSortedRowModel(),
    });

    const handleSort = (col) => {
        if (col === "sno") return;

        if (sortBy === col) {
            setOrder(order === "asc" ? "desc" : "asc");
        } else {
            setSortBy(col);
            setOrder("asc");
        }
    };

    const totalPages = Math.ceil(total / limit);

    return (
        <div className="data-table" style={{ padding: "20px", textAlign: "center" }}>
            <h2>Product Table (Server-side Search, Sorting, Pagination)</h2>


            <div className="search-container">
                <img src="/icons/search.svg" alt="search" className="search-icon" />
                <input
                    type="text"
                    placeholder="Search by title..."
                    value={search}
                    onChange={(e) => {
                        setSearch(e.target.value);
                        setPage(1);
                    }}
                    className="search-input"
                />
            </div>


            <table className="data-table" border="1" cellPadding="10">
                <thead>
                    {table.getHeaderGroups().map((headerGroup) => (
                        <tr key={headerGroup.id}>
                            {headerGroup.headers.map((header) => {
                                const key = header.column.columnDef.accessorKey;
                                const alignClass =
                                    key === "price" ? "price" :
                                        key === "sno" ? "center" : "";

                                return (
                                    <th
                                        key={header.id}
                                        className={alignClass}
                                        onClick={() =>
                                            key !== "sno" && handleSort(key)
                                        }
                                        style={{
                                            cursor: key === "sno" ? "default" : "pointer",
                                        }}
                                    >
                                        {flexRender(header.column.columnDef.header, header.getContext())}

                                        {key !== "sno" &&
                                            sortBy === key &&
                                            (order === "asc" ? (
                                                <img
                                                    src="/icons/up.svg"
                                                    alt="asc"
                                                    style={{
                                                        width: "14px",
                                                        height: "14px",
                                                        marginLeft: "5px",
                                                        verticalAlign: "middle",
                                                    }}
                                                />
                                            ) : (
                                                <img
                                                    src="/icons/down.svg"
                                                    alt="desc"
                                                    style={{
                                                        width: "14px",
                                                        height: "14px",
                                                        marginLeft: "5px",
                                                        verticalAlign: "middle",
                                                    }}
                                                />
                                            ))}
                                    </th>
                                );
                            })}
                        </tr>
                    ))}
                </thead>

                <tbody>
                    {table.getRowModel().rows.length > 0 ? (
                        table.getRowModel().rows.map((row) => (
                            <tr key={row.id}>
                                {row.getVisibleCells().map((cell) => {
                                    const key = cell.column.columnDef.accessorKey;
                                    const alignClass =
                                        key === "price" ? "price" :
                                            key === "sno" ? "center" : "";

                                    return (
                                        <td key={cell.id} className={alignClass}>
                                            {flexRender(cell.column.columnDef.cell, cell.getContext())}
                                        </td>
                                    );
                                })}
                            </tr>
                        ))
                    ) : (
                        <tr>
                            <td colSpan={columns.length}>No data found</td>
                        </tr>
                    )}
                </tbody>
            </table>


            <div className="pagination-container">
                <button
                    className="pagination-btn"
                    onClick={() => setPage((prev) => Math.max(prev - 1, 1))}
                    disabled={page === 1}
                >
                    <img
                        src="/icons/back.svg"
                        alt="Previous"
                        className="pagination-icon"
                    />
                    Prev
                </button>

                <span className="pagination-info">
                    Page {page} of {totalPages || 1}
                </span>

                <button
                    className="pagination-btn"
                    onClick={() => setPage((prev) => (prev < totalPages ? prev + 1 : prev))}
                    disabled={page >= totalPages}
                >
                    Next
                    <img
                        src="/icons/forward.svg"
                        alt="Next"
                        className="pagination-icon"
                    />
                </button>

                <select
                    value={limit}
                    onChange={(e) => {
                        setLimit(Number(e.target.value));
                        setPage(1);
                    }}
                    className="pagination-select"
                >
                    {[5, 10, 20, 50].map((num) => (
                        <option key={num} value={num}>
                            {num} / page
                        </option>
                    ))}
                </select>
            </div>
        </div>
    );
};

export default TanStack;
